<?php

namespace App\Http\Controllers\Admin;

use App\Education;
use App\Http\Requests\EducationRequest;
use Exception;
use App\EducationDescription;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;

class EducationController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/educations';
    /** View folder */
    private $viewDirectory = 'admin.educations.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('educations.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the courses.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Education::join('education_descriptions AS cd', 'educations.id', 'cd.education_id')
            ->where('cd.language_id', currentLanguage()->id)
            ->select(['cd.name', 'educations.*']);

        if ($request->date_from) {
            $query->whereDate('educations.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('educations.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('cd.name', 'LIKE', '%'.$request->name.'%');
        }


        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $educations= $query->paginate(100);

        return view($this->viewDirectory.'grid',get_defined_vars());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('educations.store');
        $head = metaFields('educations', 'new', getCurrentLocale()) ?? __('educations.new');
        $title = $head;
        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     * @param EducationRequest $request
     * @return JsonResponse
     */
    public function store(EducationRequest $request) :JsonResponse
    {
        $education = Education::create();
        $this->saveData( $request, $education->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('educations.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param Education $education
     * @return View
     */
    public function edit(Education $education) :View
    {
        $action= route('educations.update', $education->id);
        $head = metaFields('educations', 'edit', getCurrentLocale()) ?? __('educations.edit');
        $title = $head;
        $query = EducationDescription::where('education_id', $education->id)
            ->join('languages', 'languages.id', 'education_descriptions.language_id')
            ->select(['education_descriptions.*', 'languages.local']);

        $educationDescription= $query->get();

        foreach ($educationDescription as $row) {
            $education[$row->local]= $row;
        }

        $local = getCurrentLocale();

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     * @param EducationRequest $request
     * @param Education $education
     * @return JsonResponse
     */
    public function update(EducationRequest $request, Education $education) :JsonResponse
    {
        $education->touch();
        // Delete old description
        EducationDescription::where('education_id', $education->id)->delete();
        $this->saveData( $request, $education->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('educations.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Handle Save form data
     *
     * @param EducationRequest $request
     * @param int $education_id
     * @return void
     */

    private function saveData(EducationRequest $request, int $education_id ):void
    {
        $requestData= $request->all();
        foreach (languages() as $lang) {
            $data=[
                'name'=> $requestData['name_'.$lang->local],
                'education_id'=> $education_id,
                'language_id'=> $lang->id,
            ];
            EducationDescription::create($data);
        }

    }



    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $education = Education::withTrashed()->find($id);
        if ($education) {
            if ($education->deleted_at) {
                $education->forceDelete();
            } else {
                $education->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several courses by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            Education::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Education::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $education = Education::withTrashed()->find($id);

        if ($education){
            $education->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several courses by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $educations = Education::whereIn('id', $request->ids)->onlyTrashed();
        if ($educations){
            $educations->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
