<?php

namespace App\Http\Controllers\Admin;


/** Ajax Request */
// use App\Http\Resources\CategoryResource;


use Exception;
use App\VideoCategory;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\VideoCategoryDescription;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Redirector;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\VideoCategoryRequest;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class VideoCategoryController extends Controller
{
    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/videoCategories';
    /** View folder */
    private $viewDirectory = 'admin.video_categories.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the video categories.
     *
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('videoCategories.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the video categories.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = VideoCategory::latest()
        ->join('video_category_descriptions AS cd', 'video_categories.id', 'cd.video_category_id')
        ->join('languages', 'languages.id', 'cd.language_id')
        ->where('local', LaravelLocalization::getCurrentLocale())
        ->select(['cd.name', 'video_categories.*']);

        if ($request->date_from) {
            $query->whereDate('video_categories.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('video_categories.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('cd.name', 'LIKE', '%'.$request->name.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }
        $videoCategories= $query->paginate(100);

        return view($this->viewDirectory.'grid',compact('videoCategories'));
    }

    /**
     * Show the form for creating a new video category.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('videoCategories.store');
        $head = metaFields('videoCategories', 'new', getCurrentLocale()) ?? __('videoCategories.new');
        $title = $head;
        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created video category in storage.
     *
     * @param VideoCategoryRequest $request
     * @return RedirectResponse|Redirector
     */
    public function store(VideoCategoryRequest $request) :RedirectResponse
    {
        $category = VideoCategory::create($request->all());

        // Insert video category Description
        $this->saveData( $request, $category->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Show the form for editing the specified category.
     *
     * @param VideoCategory $videoCategory
     * @return View
     */
    public function edit(VideoCategory $videoCategory) :View
    {
        $action= route('videoCategories.update', $videoCategory->id);
        $head = metaFields('videoCategories', 'edit', getCurrentLocale()) ?? __('videoCategories.edit');
        $title = $head;

        $query = VideoCategoryDescription::where('video_category_id', $videoCategory->id)
        ->join('languages', 'languages.id', 'video_category_descriptions.language_id')
        ->select(['video_category_descriptions.*', 'languages.local']);

        $videoCategoryDescription= $query->get();

        foreach ($videoCategoryDescription as $row) {
            $videoCategory[$row->local]= $row;
        }
        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified video category in storage.
     *
     * @param VideoCategoryRequest $request
     * @param VideoCategory $videoCategory
     * @return RedirectResponse|Redirector
     */
    public function update(VideoCategoryRequest $request, VideoCategory $videoCategory) :RedirectResponse
    {

        $videoCategory->update(['image' => $request->image ?? null]);

        // Delete old video category description
        VideoCategoryDescription::where('video_category_id', $videoCategory->id)->delete();

        // Insert new video category description
        $this->saveData( $request, $videoCategory->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param VideoCategoryRequest $request
     * @param int $video_category_id
     * @return void
     */

     private function saveData(VideoCategoryRequest $request, int $video_category_id ) :void
     {
        $requestData= $request->all();

        foreach (languages() as $lang) {
            $data=[
                'video_category_id'=> $video_category_id,
                'language_id'=> $lang->id,
                'name'=> $requestData['name_'.$lang->local],
                'slug'=> $requestData['slug_'.$lang->local],
                'keywords'=> $requestData['keywords_'.$lang->local],
                'meta_description'=> $requestData['meta_description_'.$lang->local],
            ];

            VideoCategoryDescription::create($data);
        }

     }

    /**
     * Remove the specified video category from storage.
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $videoCategory= VideoCategory::withTrashed()->find($id);
        if ($videoCategory) {
            if ($videoCategory->deleted_at) {
                $videoCategory->forceDelete();
            } else {
                $videoCategory->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several video categories by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            VideoCategory::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            VideoCategory::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified video category from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $VideoCategory= VideoCategory::withTrashed()->find($id);
        if ($VideoCategory) {
            $VideoCategory->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);

    }

    /**
     * Restore several categories by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        VideoCategory::whereIn('id', $ids)->onlyTrashed()->restore();
        return response()->json(['message'=> __('dashboard.saveDone')]);
    }
}
