<?php

namespace App\Http\Controllers\Admin;


use App\VisitorMessageReply;
use Exception;
use App\VisitorMessage;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use Illuminate\Http\Request;
use App\Jobs\VisitorMessagesJob;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Redirector;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\VisitorMessageRequest;

class VisitorMessageController extends Controller
{
    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/visitorMessages';
    /** View folder */
    private $viewDirectory = 'admin.visitor_messages.';


    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }


    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('visitorMessages.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the newsletters.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = VisitorMessage::latest();

        if ($request->date_from) {
            $query->whereDate('visitor_messages.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('visitor_messages.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('visitor_messages.name', 'LIKE', '%'.$request->name.'%');
        }

        if ($request->phone) {
            $query->where('visitor_messages.phone', 'LIKE', '%'.$request->phone.'%');
        }
        if ($request->email) {
            $query->where('visitor_messages.email', 'LIKE', '%'.$request->email.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $visitorMessages= $query->paginate(10);

        return view($this->viewDirectory.'grid', get_defined_vars());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('visitorMessages.store');
        $head = metaFields('visitorMessages', 'new', getCurrentLocale()) ?? __('visitorMessages.edit');
        $title = $head;
        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param VisitorMessageRequest $request
     * @return JsonResponse
     */
    public function store(VisitorMessageRequest $request) :JsonResponse
    {
        $this->saveData($request);
        session()->flash('message', __('visitorMessages.email_success'));
        return response()->json(['url' => route('visitorMessages.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('visitorMessages.email_success'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param VisitorMessage $visitorMessage
     * @return View
     */
    public function edit(VisitorMessage $visitorMessage) :View
    {
        $head = metaFields('visitorMessages', 'new', getCurrentLocale()) ?? __('visitorMessages.edit');
        $title = $head;
        $replies = $visitorMessage->replies;
        $visitorMessage->update(['read' => 1]);
        $action= route('visitorMessages.update', $visitorMessage->id);
        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param VisitorMessageRequest $request
     * @param VisitorMessage $visitorMessage
     * @return JsonResponse
     */
    public function update(VisitorMessageRequest $request, VisitorMessage $visitorMessage) :JsonResponse
    {
        $this->saveData($request, $visitorMessage);
        session()->flash('message', __('visitorMessages.email_success'));
        return response()->json(['url' => route('visitorMessages.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('visitorMessages.email_success'));
    }

    /**
     * Handle Save form data
     *
     * @param VisitorMessageRequest $request
     * @param $visitorMessage
     * @return void
     */

    private function saveData(VisitorMessageRequest $request, $visitorMessage = null):void
    {
        $data= $request->all();
        $data['to'] = $data['visitor_email'];
        $this->dispatch(new VisitorMessagesJob($data, $this->viewDirectory.'email'));

        VisitorMessageReply::create([
            'reply' => $request->reply,
            'visitor_email' => $request->visitor_email,
            'user_id' => Auth::id(),
            'visitor_message_id' => $visitorMessage ? $visitorMessage->id : null,
        ]);

        $visitorMessage ? $visitorMessage->update(['reply' => 1]) : null;
    }

    public function details()
    {
        $head=  __('visitorMessages.page_details');
        $detail = VisitorMessageDetail::first();
        $descriptions = VisitorMessageDetailDescription::where('visitor_message_detail_id', $detail->id)
            ->join('languages', 'languages.id', 'visitor_message_detail_descriptions.language_id')
            ->select(['visitor_message_detail_descriptions.*', 'languages.local'])->get();

        foreach ($descriptions as $description) {
            $detail[$description->local] = $description;
        }

        $action= route('visitorMessages.updateDetails', $detail->id);

        return view($this->viewDirectory.'details', get_defined_vars());
    }

    public function updateDetails(VisitorMessageRequest $request, VisitorMessageDetail $visitorMessageDetail)
    {
        $data = $request->all();
        $visitorMessageDetail->update($data);
        VisitorMessageDetailDescription::where('visitor_message_detail_id', $visitorMessageDetail->id)->delete();

        foreach (languages() as $lang) {
            $detailData=[
                'visitor_message_detail_id'=> $visitorMessageDetail->id,
                'language_id'=> $lang->id,
                'header_title'=> $data['header_title_'.$lang->local],
                'title'=> $data['title_'.$lang->local],
            ];
            VisitorMessageDetailDescription::create($detailData);
        }

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));


    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $message = VisitorMessage::withTrashed()->find($id);
        if ($message) {
            if ($message->deleted_at) {
                $message->forceDelete();
            } else {
                $message->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            VisitorMessage::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            VisitorMessage::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $message = VisitorMessage::withTrashed()->find($id);

        if ($message){
            $message->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $messages = VisitorMessage::whereIn('id', $request->ids)->onlyTrashed();
        if ($messages){
            $messages->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
