<?php

namespace App\Http\Controllers\Admin;

use App\WorkHour;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\WorkHourRequest;
use App\Http\Controllers\Controller;

class WorkHourController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/workHours';
    /** View folder */
    private $viewDirectory = 'admin.work_hours.';

    private $weekDays = [];

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;

        \Carbon\Carbon::setWeekStartsAt(\Carbon\Carbon::FRIDAY);
        $date = \Illuminate\Support\Carbon::now();

        $weekDays = [];
        $startOfWeek = $date->startOfWeek();

        for ($i = 1; $i <= 7; $i++){
            $x = $startOfWeek->addDay();
            $key = strtolower($x->shortEnglishDayOfWeek);
            $day = $x->getTranslatedDayName();
            $weekDays[$key] = $day;
        }

        $this->weekDates = $weekDays;

    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('workHours.head');
        $request= $request->toArray();
        $weekDays = $this->weekDates;

        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the workHours.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $langId = currentLanguage()->id;
        $query = WorkHour::orderBy('work_hours.id');

        if ($request->date_from) {
            $query->whereDate('work_hours.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('work_hours.created_at', '<=', $request->date_to);
        }

        if ($request->week_day) {
            $query->where('work_hours.week_day', $request->week_day);
        }

        if ($request->time_from) {
            $query->where('work_hours.time_from', 'LIKE', '%'.$request->time_from.'%');
        }

        if ($request->time_to) {
            $query->where('work_hours.time_to', 'LIKE', '%'.$request->time_to.'%');
        }


        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $workHours= $query->paginate(100);

        $weekDays = $this->weekDates;


        return view($this->viewDirectory.'grid', get_defined_vars());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('workHours.store');
        $head = metaFields('workHours', 'new', getCurrentLocale()) ?? __('workHours.new');
        $title = $head;
        $weekDays = $this->weekDates;

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param WorkHourRequest $request
     * @return JsonResponse
     */
    public function store(WorkHourRequest $request) :JsonResponse
    {
        WorkHour::create($request->all());
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('workHours.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param WorkHour $workHour
     * @return View
     */
    public function edit(WorkHour $workHour) :View
    {
        $action= route('workHours.update', $workHour->id);
        $head = metaFields('workHours', 'edit', getCurrentLocale()) ?? __('workHours.edit');
        $title = $head;

        $weekDays = [$workHour->week_day => $this->weekDates[$workHour->week_day]];

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param WorkHourRequest $request
     * @param WorkHour $workHour
     * @return JsonResponse
     */
    public function update(WorkHourRequest $request, WorkHour $workHour) : JsonResponse
    {
        $workHour->update($request->all());
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('workHours.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

}
