<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed video_type
 * @property mixed video_url
 * @property mixed images
 */
class DirectorRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('directors'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [];

        foreach (languages() as $lang) {
            $lang_rules=[
                'name_'.$lang->local=> 'required|min:3|max:100',
                'specialist_'.$lang->local=> 'required|min:3|max:100',
//                'description_'.$lang->local=> 'required|min:3|max:600',
            ];
            $rules= array_merge($rules, $lang_rules);
        }

        $rules['image'] = 'required';
        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('directors');
        $local = getCurrentLocale();
        $attributes=[
            'image' =>  $field[$local]['image'] ?? __('directors.image'),
        ];


        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? ' (عربي) ' : ' (English) ';
            $lang_attributes=[
                'name_'.$lang->local => ($field[$lang->local]['name'] ?? __('directors.name')) . $local,
                'specialist_'.$lang->local => ($field[$lang->local]['specialist'] ?? __('directors.specialist')) . $local,
//                'description_'.$lang->local => ($field[$lang->local]['description'] ?? __('directors.description')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            parent::failedValidation($validator);
        }
    }
}
