<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed images
 */
class PhotoCategoryRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('photoCategories'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules=[
            'images' => 'required',
//            'image_title_'.getCurrentLocale().'.*' => 'required|min:3',
            'photo_category_id' => 'sometimes',
            'photo_id' => 'sometimes',
            'name_'.getCurrentLocale()=> 'required|min:3',
            'slug_'.getCurrentLocale()=> [
                "required",
                'min:3',
                Rule::unique('photo_category_descriptions', 'slug')->ignore($this->photoCategory->id, 'photo_category_id')
            ],
        ];

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('photoCategories');
        $local = getCurrentLocale();
        $attributes=[
            'name_'.$local=> $field[$local]['name'] ??  __('photoCategories.name'),
            'slug_'.$local=> $field[$local]['slug'] ??  __('photoCategories.slug'),
            'images' => $field[$local]['images'],
        ];
        if (request()->has('images')){
            foreach (request('images') as $key => $image) {
                $imagesAttributes=[
                    'image_title_'.$local.'.'.$key=> $field[$local]['image_title'] ??  __('photoCategories.title'),

                ];
                $attributes= array_merge($attributes, $imagesAttributes);

            }
        }

        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
