<?php

namespace App\Http\Requests;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed status
 */
class SettingRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('settings'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $phoneRegex = '/^([\+00\-]?[0-9]{1,3}[0-9\s]{8,16})$/';
        $rules = [];
        foreach (languages() as $lang) {
            $lang_rules=[
                'website_name_'.$lang->local=> 'required|min:3',
                'site_address_'.$lang->local=> 'required|min:3',
                'Keywords_'.$lang->local=> 'required|min:3',
                'meta_description_'.$lang->local=> 'required|min:3',
                'special_num_1_title_'.$lang->local=> 'required|min:3',
                'special_num_2_title_'.$lang->local=> 'required|min:3',
                'special_num_3_title_'.$lang->local=> 'required|min:3',
                'special_num_4_title_'.$lang->local=> 'required|min:3',
            ];
            $rules= array_merge($rules, $lang_rules);
        }

        $fieldRules = [
            'site_email' => 'required|email',
            'site_phone' => 'required|required|regex:'.$phoneRegex,
            'site_tel' => 'required|required',
            'mail_driver' => 'required',
            'mail_host' => 'required',
            'mail_port' => 'required|numeric',
            'mail_username' => 'required',
            'mail_password' => 'required',
            'mail_encryption' => 'required',
            'mail_from_address' => 'required',
            'mail_from_name' => 'required',
            'website_logo' => 'sometimes',
            'facebook' => 'nullable',
            'twitter' => 'nullable',
//            'tiktok' => 'required',
            'instagram' => 'nullable',
            'snapchat' => 'nullable',
            'linkedin' => 'nullable',
            'whatsapp' => 'nullable',
            'site_map' => 'required',
            'special_num_1_value' => 'required|numeric',
            'special_num_2_value' => 'required|numeric',
            'special_num_3_value' => 'required|numeric',
            'special_num_4_value' => 'required|numeric',
            'show_start_pop_up' => 'required|in:1,0',
            'booking_appointment_url' => 'required|url',
//            'snapchat' => 'required',
//            'career_link' => 'required',
            'website_disable' => 'sometimes',
            'dashboard_disable' => 'sometimes',
        ];

        $rules = array_merge($rules, $fieldRules);




        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('settings');
        $local = getCurrentLocale();

        $attributes =  [
            'site_email' => $field[$local]['site_email'] ?? __('settings.site_email'),
            'site_phone' => $field[$local]['site_phone'] ?? __('settings.site_phone'),
            'site_tel' => $field[$local]['site_tel'] ?? __('settings.site_tel'),
            'mail_driver' => $field[$local]['mail_driver'] ?? __('settings.mail_driver'),
            'mail_host' => $field[$local]['mail_host'] ?? __('settings.mail_host'),
            'mail_port' => $field[$local]['mail_port'] ?? __('settings.mail_port'),
            'mail_username' => $field[$local]['mail_username'] ?? __('settings.mail_username'),
            'mail_password' => $field[$local]['mail_password'] ?? __('settings.mail_password'),
            'mail_encryption' => $field[$local]['mail_encryption'] ?? __('settings.mail_encryption'),
            'mail_from_address' => $field[$local]['mail_from_address'] ?? __('settings.mail_from_address'),
            'mail_from_name' => $field[$local]['mail_from_name'] ?? __('settings.mail_from_name'),
            'website_logo' => $field[$local]['website_logo'] ?? __('settings.website_logo'),
            'facebook' => $field[$local]['facebook'] ?? __('settings.facebook'),
            'tiktok' => $field[$local]['tiktok'] ?? __('settings.tiktok'),
            'instagram' => $field[$local]['instagram'] ?? __('settings.instagram'),
            'snapchat' => $field[$local]['snapchat'] ?? __('settings.snapchat'),
            'linkedin' => $field[$local]['linkedin'] ?? __('settings.linkedin'),
            'whatsapp' => $field[$local]['whatsapp'] ?? __('settings.whatsapp'),
            'career_link' => $field[$local]['career_link'] ?? __('settings.career_link'),
            'site_map' => $field[$local]['site_map'] ?? __('settings.site_map'),
            'special_num_1_value' => $field[$local]['special_num_1_value'] ?? __('settings.special_num_1_value'),
            'special_num_2_value' => $field[$local]['special_num_2_value'] ?? __('settings.special_num_2_value'),
            'special_num_3_value' => $field[$local]['special_num_3_value'] ?? __('settings.special_num_3_value'),
            'special_num_4_value' => $field[$local]['special_num_4_value'] ?? __('settings.special_num_4_value'),
            'show_start_pop_up' => $field[$local]['show_start_pop_up'] ?? __('settings.show_start_pop_up'),
            'booking_appointment_url' => $field[$local]['booking_appointment_url'] ?? __('settings.booking_appointment_url'),
        ];

        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? ' (عربي) ' : ' (English) ';

            $lang_attributes=[
                'website_name_'.$lang->local=> ($field[$lang->local]['website_name'] ?? __('settings.website_name')) . $local,
                'keywords_'.$lang->local=> ($field[$lang->local]['keywords'] ?? __('settings.keywords')) . $local,
                'meta_description_'.$lang->local=> ($field[$lang->local]['meta_description'] ?? __('settings.meta_description')) . $local,
                'special_num_1_title_'.$lang->local=> ($field[$lang->local]['special_num_1_title'] ?? __('settings.special_num_1_title')) . $local,
                'special_num_2_title_'.$lang->local=> ($field[$lang->local]['special_num_2_title'] ?? __('settings.special_num_2_title')) . $local,
                'special_num_3_title_'.$lang->local=> ($field[$lang->local]['special_num_3_title'] ?? __('settings.special_num_3_title')) . $local,
                'special_num_3_title_'.$lang->local=> ($field[$lang->local]['special_num_3_title'] ?? __('settings.special_num_3_title')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }

        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
