<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed videoCategory
 */
class VideoCategoryRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('videoCategories'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules=['image' => 'sometimes'];
        foreach (languages() as $lang) {
            $lang_rules=[
                'name_'.$lang->local=> 'required|min:3',
                'slug_'.$lang->local=> [
                    "required",
                    Rule::unique('video_category_descriptions', 'slug')->ignore($this->videoCategory? $this->videoCategory->id : 0, 'video_category_id')
                ],
                'keywords_'.$lang->local=> 'required|min:3',
                'meta_description_'.$lang->local=> 'required|min:3',
            ];
            $rules= array_merge($rules, $lang_rules);
        }
        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('videoCategories');
        $attributes=[];
        foreach (languages() as $lang) {
            $lang_attributes=[
                'name_'.$lang->local=> $field[$lang->local]['name'] ?? __('categories.name'),
                'slug_'.$lang->local=>  $field[$lang->local]['slug'] ?? __('categories.slug'),
                'keywords_'.$lang->local=> $field[$lang->local]['keywords'] ?? __('categories.keywords'),
                'meta_description_'.$lang->local=> $field[$lang->local]['meta_description'] ?? __('categories.meta_description'),
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }

    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
