<?php

namespace Tests\Feature;

use App\Photo;
use App\PhotoDescription;
use App\User;
use Tests\TestCase;
use App\PhotoCategoryDescription;
use Illuminate\Support\Facades\Session;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\DatabaseTransactions;

class PhotoTest extends TestCase
{
    // To reset DB to it's status before run test
    use DatabaseTransactions;

    // To create fake sentences to help us test and validation
    use WithFaker;

    private $user;


    protected function setUp() :void
    {
        parent::setUp();

        // to see description
        $this->withoutExceptionHandling();

        // Start session to enables csrf_token()
        Session::start();

        // Authenticate user
        $this->user = factory(User::class)->create(['role'=>'admin']);

        $this->actingAs($this->user);
    }

    /**
     * Method to ensure that the user can access to photo.
     *
     * @return void
     */
    public function testUserCanAccessPhotoPage() :void
    {

        // Photo route which visit by user
        $response = $this->get(route('photoCategories.index'));

        // Should be return status 200
        $response->assertStatus(200);
    }

    /**
     * Method to ensure that the user can read all photos.
     *
     * @return void
     */
    public function testUserCanReadAllPhotos() :void
    {
        //Given we have photo in the database
        $photoCategoryDescription = factory(PhotoCategoryDescription::class)->create();

        //When user visit the photo category
        $response = $this->get(route('photoCategories.grid'));
        // status should be 200
        $response->assertStatus(200);

        //He should be able to read the photo category
        $response->assertSee($photoCategoryDescription->name);

    }

    /**
     * Method to ensure that the create form route exists.
     *
     * @return void
     */

    public function testUserCanCreatePhoto()
    {
        //When user visit the new photo form route
        $response = $this->get(route('photoCategories.create'));

        //He should be able to see the fields which enable him to add new photo
        $response->assertStatus(200);
        $name = metaFields('photoCategories', 'name', getCurrentLocale());
        $response->assertSee($name ?? __('photoCategories.name'));

    }

    /**
     * Method to ensure that the update form route exists.
     *
     * @return void
     */

    public function testUserCanEditPhoto()
    {
        //Given we have photo in the database
        $photoCategoryDescription = factory(PhotoCategoryDescription::class)->create();

        //When user visit the photo form route
        $response = $this->get(route('photoCategories.edit', $photoCategoryDescription->photo_category_id));

        //He should be able to see the fields which enable him to edit the photo
        $response->assertStatus(200);
        $name = metaFields('photoCategories', 'name', getCurrentLocale());
        $response->assertSee($name ?? __('photoCategories.name'));

        $response->assertSee($photoCategoryDescription['name_'.getCurrentLocale()]);
    }

    /**
     * Method to ensure that the user can add photo and photo description.
     *
     * @return void
     */
    public function testUserCanAddPhoto() :void
    {
        $photoCategoryDescription = factory(PhotoCategoryDescription::class)->make();

        $dataToSave = [

            'images' => [
                '/uploads/images/5eafffbad512e.jpg',
                '/uploads/images/5eafffbad512e.jpg',
                '/uploads/images/5eafffbad512e.jpg',
            ],

            'image_title_'.getCurrentLocale() => [
                $this->faker->sentence,
                $this->faker->sentence,
                $this->faker->sentence
            ],

            'photo_category_id' => $photoCategoryDescription->photo_category_id,
            'name_'.getCurrentLocale() => $this->faker->sentence,
            'slug_'.getCurrentLocale() => $this->faker->unique()->slug(1),
        ];


        //When user submits post request to create photo endpoint
        $response= $this->post(route('photoCategories.store'), array_merge( $dataToSave, ['_token'=> csrf_token()]));

        // The redirect response header status is 302
        $response->assertStatus(302);

        // The response redirect to photos
        $response->assertRedirect('admin/photoCategories');


        //It gets stored in the database
        foreach ($dataToSave['images'] as $image){
            $this->assertDatabaseHas('photos', ['image'=> $image]);
        }


        foreach ($dataToSave['image_title_'.getCurrentLocale()] as $title){
            $this->assertDatabaseHas('photo_descriptions',
                ['title'=> $title, 'language_id'=> currentLanguage()->id]
            );
        }


        $this->assertDatabaseHas('photo_category_descriptions',
            [
                'name'=> $dataToSave['name_'.getCurrentLocale()],
                'slug'=> $dataToSave['slug_'.getCurrentLocale()],
                'language_id'=> currentLanguage()->id,
            ]
        );


        // Session success message
        $response->assertSessionHas('message', __('dashboard.saveDone'));

        // Page appears in the Pages panel
        $response = $this->get(route('photoCategories.grid'));

        //He should be able to read the photo
        $response->assertSee($dataToSave['name_'.getCurrentLocale()]);

    }

    /**
     * Method to ensure that the user can Edit photo.
     *
     * @return void
     */
    public function testUserCanUpdatePhoto() :void
    {
        $photoCategoryDescription = factory(PhotoCategoryDescription::class)->make();

        $dataToSave = [

            'images' => [
                '/uploads/images/5eafffbad512e.jpg',
                '/uploads/images/5eafffbad512e.jpg',
                '/uploads/images/5eafffbad512e.jpg',
            ],

            'image_title_'.getCurrentLocale() => [
                $this->faker->sentence,
                $this->faker->sentence,
                $this->faker->sentence
            ],

            'photo_category_id' => $photoCategoryDescription->photo_category_id,
            'name_'.getCurrentLocale() => $this->faker->sentence,
            'slug_'.getCurrentLocale() => $this->faker->unique()->slug(1),
        ];

        //When user submits post request to edit photo endpoint
        $response= $this->put(route('photoCategories.update', $photoCategoryDescription->photo_category_id),array_merge( $dataToSave, ['_token'=> csrf_token()] ) );

        // The redirect response header status is 302
        $response->assertStatus(302);

        // The response redirect to photos
        $response->assertRedirect('admin/photoCategories');


        //It gets stored in the database
        foreach ($dataToSave['images'] as $image){
            $this->assertDatabaseHas('photos', ['image'=> $image]);
        }


        foreach ($dataToSave['image_title_'.getCurrentLocale()] as $title){
            $this->assertDatabaseHas('photo_descriptions',
                ['title'=> $title, 'language_id'=> currentLanguage()->id]
            );
        }


        $this->assertDatabaseHas('photo_category_descriptions',
            [
                'name'=> $dataToSave['name_'.getCurrentLocale()],
                'slug'=> $dataToSave['slug_'.getCurrentLocale()],
                'language_id'=> currentLanguage()->id,
            ]
        );


        // Session success message
        $response->assertSessionHas('message', __('dashboard.saveDone'));

        // Page appears in the Page grid
        $response = $this->get(route('photoCategories.grid'));

        //He should be able to read the photo
        $response->assertSee($dataToSave['name_'.getCurrentLocale()]);
    }

    /**
     * Method to ensure that the user can send the photo to trash.
     *
     * @return void
     */
    public function testPhotosTrash() :void
    {

        //Add a photo object
        $photoCategoryDescription = factory(PhotoCategoryDescription::class)->make();

        //When the user hit's with endpoint to delete the photo
        $this->delete(route('photoCategories.destroy', $photoCategoryDescription->photo_category_id), ['_token'=> csrf_token()]);

        //The photo should be deleted from the database.
        $this->assertSoftDeleted('photo_categories',['id'=> $photoCategoryDescription->photo_category_id]);
    }

    /**
     * Method to ensure that the user can delete the photo from database.
     *
     * @return void
     */
    public function testPhotosDelete() :void
    {
        //Add a photo object
        $photoCategoryDescription = factory(PhotoCategoryDescription::class)->make();

        // user sent the photo to trash first, as he can not delete it from the first click
        $this->delete(route('photoCategories.destroy', $photoCategoryDescription->photo_category_id), ['_token'=> csrf_token()]);

        //When the user hit's the endpoint to delete the photo
        $this->delete(route('photoCategories.destroy', $photoCategoryDescription->photo_category_id), ['_token'=> csrf_token()]);

        //The photo should be deleted from the database.

        $this->assertDatabaseMissing('photo_categories',['id'=> $photoCategoryDescription->photo_category_id]);

        $this->assertDatabaseMissing('photo_category_descriptions',['photo_category_id'=> $photoCategoryDescription->photo_category_id]);
    }

    /**
     * Method to ensure that the user can send multiple photos to trash.
     *
     * @return void
     */
    public function testPhotosMultiTrash() :void
    {
        //Add a photo object
        $photoCategoryDescription = factory(PhotoCategoryDescription::class, 3)->make();

        $ids= $photoCategoryDescription->pluck('photo_category_id')->toArray();

        //When the user hit's the endpoint to send the photo to trash
        $this->delete(route('photoCategories.destroyAll'), ['_token'=> csrf_token(), 'ids'=> $ids]);
        //The photo should be deleted from the database.
        $this->assertSoftDeleted('photo_categories',['id'=> $ids]);
    }

    /**
     * Method to ensure that the user can delete multiple photo categories.
     *
     * @return void
     */
    public function testPhotosMultiDelete() :void
    {
        //Add a photo object
        $photoCategoryDescription = factory(PhotoCategoryDescription::class, 3)->make();

        $ids= $photoCategoryDescription->pluck('photo_category_id')->toArray();

        //When the user hit's the endpoint to send the photos to trash
        $this->delete(route('photoCategories.destroyAll'), ['_token'=> csrf_token(), 'ids'=> $ids]);

        //When the user hit's the endpoint to delete the photos from the db
        $this->delete(route('photoCategories.destroyAll'), ['_token'=> csrf_token(), 'ids'=> $ids, 'force' => true]);

        //The photo category should be deleted from the database.
        $this->assertDatabaseMissing('photo_categories',['id'=> $ids]);

        $this->assertDatabaseMissing('photo_category_descriptions',['photo_category_id'=> $ids]);

    }

    /**
     * Method to ensure that the user can restore the photo from trash.
     *
     * @return void
     */
    public function testPhotosRestore() :void
    {
        //Add a photo object
        $photoCategoryDescription = factory(PhotoCategoryDescription::class)->make();

        //the user send the photo to trash
        $this->delete(route('photoCategories.destroy', $photoCategoryDescription->photo_category_id), ['_token'=> csrf_token()]);

        $this->assertSoftDeleted('photo_categories',['id'=> $photoCategoryDescription->photo_category_id]);

        //the user restore the photo
        $this->put(route('photoCategories.restore', $photoCategoryDescription->photo_category_id), ['_token'=> csrf_token()]);

        //The photo should be restored .
        $this->assertDatabaseHas('photo_categories',['id'=> $photoCategoryDescription->photo_category_id, 'deleted_at'=> null]);
    }

    /**
     * Method to ensure that the user can restore multiple photos.
     *
     * @return void
     */
    public function testPhotosMultiRestore() :void
    {
        //Add a photo object
        $photoCategoryDescription = factory(PhotoCategoryDescription::class, 3)->make();

        $ids= $photoCategoryDescription->pluck('photo_category_id')->toArray();

        //When the user hit's the endpoint to send the photo to trash
        $response = $this->delete(route('photoCategories.destroyAll'), ['_token' => csrf_token(), 'ids' => $ids]);

        // Test last one to ensure soft deleted process done successfully
        $this->assertSoftDeleted('photo_categories', ['id' => $ids]);

        //When the user hit's the endpoint to restore the photos from the trash
        $this->put(route('photoCategories.restoreAll'), ['_token' => csrf_token(), 'ids' => $ids]);

        // Test last one to ensure restore process done successfully
        $this->assertDatabaseHas('photo_categories', ['id' => $ids, 'deleted_at' => null]);

    }
}
